<?php

namespace App\Http\Repositories\Eloquent\Admin;

use App\Models\Restaurant;
use App\Models\RestaurantOrder;
use Carbon\Carbon;
use DevxPackage\AbstractRepository;

class RestaurantRepository extends AbstractRepository
{

    protected $model;
    protected $restaurantOrder;

    public function __construct(Restaurant $model, RestaurantOrder $restaurantOrder)
    {
        $this->model = $model;
        $this->restaurantOrder = $restaurantOrder;
    }

    public function crudName(): string
    {
        return 'restaurants';
    }

    public function index($offset, $limit)
    {
        return $this->pagination($offset, $limit);
    }

    public function pagination($offset, $limit)
    {
        return $this->model->with($this->model->model_relations())->withCount($this->model->model_relations_counts())->unArchive()->orderBy('id', 'DESC')->paginate(PAGINATION_COUNT);
    }

    public function create()
    {
        return view('admin.restaurants.create');
    }

    public function store($request)
    {
        $translations = [];
        $model = new $this->model();
		$model->avg_prices = (int)$request->avg_prices ?? 0;
		$model->offer = (int)$request->offer ?? 0;
		$model->map_location = $request->map_location;
		$model->country_id = $request->country_id ?? null;
		$model->owner_id = $request->owner_id ?? null;
		$model->open_all_time = $request->open_all_time ?? 0;
        foreach (config('translatable.locales') as $locale) {
            $data = [];
            if (!empty($request->input("name.$locale")))    $data['name'] = $request->input("name.$locale");
            if (!empty($request->input("info.$locale")))    $data['info'] = $request->input("info.$locale");
            if (!empty($request->input("location.$locale")))    $data['location'] = $request->input("location.$locale");
            if (!empty($data)) {
                $translations[$locale] = $data;
            }
        }
        if (!$request->hasFile('file') == null) {
            $model->img = uploadIamge($request->file('file'), $this->crudName());
        }
        $model->save();
        foreach ($request->days as $day) {
            if (!empty($day['day_status']) || !empty($day['start']) || !empty($day['end'])) {
                $days [] = [
                    'created_at' => now(),
                    'updated_at' => now(),
                    'day' => $day['day'], 
                    'day_status' => $day['day_status'], 
                    'end' => $day['end'] ? Carbon::createFromFormat('H:i', $day['end'])->format('h:i A') : null, 
                    'start' => $day['start'] ? Carbon::createFromFormat('H:i', $day['start'])->format('h:i A') : null, 
                    'restaurant_id' => $model->id,
                ];
            }
        }
        if (count($days) > 0) {
            $model->days()->insert($days);
        }
        if (isset($request->times) && count($request->times) > 0) {
            $times = [];
            foreach ($request->times as $time) {
                $times [] = [
                    'created_at' => now(),
                    'updated_at' => now(),
                    'type' => $time['type'], 
                    'restaurant_id' => $model->id,
                    'time' => Carbon::createFromFormat('H:i', $time['time'])->format('h:i A'), 
                ];
            }
            $model->times()->insert($times);
        }
		$model->albums = isset($request['albums']) ? uploadIamges($request['albums'], $this->crudName()) : null;
        $model->update($translations);
        return $model;
    }

    public function edit($id)
    {
        $restaurant = $this->findOne($id);
        return view('admin.restaurants.update', compact('restaurant'));
    }
    
    public function update($request, $id)
    {
        $translations = [];
        $model = $this->model->where('id', $id)->first();
		$model->avg_prices = (int)$request->avg_prices ?? 0;
		$model->offer = (int)$request->offer ?? 0;
		$model->map_location = $request->map_location;
		$model->country_id = $request->country_id ?? null;
		$model->owner_id = $request->owner_id ?? $model->owner_id;
		$model->open_all_time = $request->open_all_time ?? 0;
        $model->deleteTranslations();
        foreach (config('translatable.locales') as $locale) {
            $data = [];
            if (!empty($request->input("name.$locale")))    $data['name'] = $request->input("name.$locale");
            if (!empty($request->input("info.$locale")))    $data['info'] = $request->input("info.$locale");
            if (!empty($request->input("location.$locale")))    $data['location'] = $request->input("location.$locale");
            if (!empty($data)) {
                $translations[$locale] = $data;
            }
        }
        if (!$request->hasFile('file') == null) {
            $model->img = uploadIamge($request->file('file'), $this->crudName());
        }
        if (isset($request->albums)) {
            $albums = uploadIamges($request->albums, $this->crudName());
            $model->albums = is_null($model->albums) ? $albums : implode(',', $model->albums) .','. $albums;
        }
        $model->save();
        if (isset($request->days) && count($request->days) > 0) {
            $days = [];
            foreach ($request->days as $day) {
                $days [] = [
                    'created_at' => now(),
                    'updated_at' => now(),
                    'day' => $day['day'], 
                    'day_status' => $day['day_status'], 
                    'end' => $day['end'] ? Carbon::createFromFormat('H:i', $day['end'])->format('h:i A') : null, 
                    'start' => $day['start'] ? Carbon::createFromFormat('H:i', $day['start'])->format('h:i A') : null, 
                    'restaurant_id' => $model->id,
                ];
            }
            $model->days()->delete();
            $model->days()->insert($days);
        }
        if (isset($request->times) && count($request->times) > 0) {
            $times = [];
            foreach ($request->times as $time) {
                $times [] = [
                    'created_at' => now(),
                    'updated_at' => now(),
                    'type' => $time['type'], 
                    'restaurant_id' => $model->id,
                    'time' => Carbon::createFromFormat('H:i', $time['time'])->format('h:i A'), 
                ];
            }
            $model->times()->delete();
            $model->times()->insert($times);
        }
        $model->update($translations);
        return $model;
    }

    public function archivesPage($offset, $limit)
    {
        $restaurants = $this->archives($offset, $limit);
        return view('admin.restaurants.archives', compact('restaurants'));
    }
    
    public function uploadAlbums($request, $id)
    {
        $model = $this->model->where('id', $id)->first();
        if ($request->hasFile('file')) {
            $file = uploadIamge($request->file('file'), $this->crudName());
            $oldImages = $model->albums ? explode(',', $model->albums) : [];
            $oldImages[] = $file;
            $model->albums = implode(',', $oldImages);
            $model->save();
            return response()->json(['success' => true]);
        }
        return response()->json(['error' => 'No file uploaded'], 400);
    }

    public function deleteImg($request, $id)
    {
        $model = $this->model->where('id', $id)->first();
        if ($request->input('img_name')) {
            $imageToDelete = $request->input('img_name');
            $oldImages = $model->albums ? explode(',', $model->albums) : [];
            $filteredImages = array_filter($oldImages, function($img) use ($imageToDelete) {
                return trim($img) !== ltrim(trim($imageToDelete), '/');
            });
            $model->albums = implode(',', $filteredImages);
            $model->save();
            if (file_exists(public_path($imageToDelete))) {
                unlink(public_path($imageToDelete));
            }
            return response()->json(['success' => true]);
        }
        return response()->json(['error' => 'No file uploaded'], 400);
    }

    public function orders($offset, $limit)
    {
        return $this->restaurantOrder->with($this->restaurantOrder->model_relations())->unArchive()->orderBy('id', 'DESC')->paginate(PAGINATION_COUNT);
    }

}