<?php

namespace App\Http\Repositories\Eloquent\Admin;

use App\Models\User;
use App\Models\GuideOrder;
use Carbon\Carbon;
use DevxPackage\AbstractRepository;
use Illuminate\Support\Facades\DB;

class UserRepository extends AbstractRepository
{

    protected $model;
    protected $guideOrder;

    public function __construct(User $model, GuideOrder $guideOrder)
    {
        $this->model = $model;
        $this->guideOrder = $guideOrder;
    }

    public function crudName(): string
    {
        return 'users';
    }

    public function index($offset, $limit)
    {
        $users = $this->pagination($offset, $limit);
        return view('admin.users.index', compact('users'));
    }

    public function pagination($offset, $limit)
    {
        return $this->model->with($this->model->model_relations())->withCount($this->model->model_relations_counts())->unArchive()->orderBy('id', 'DESC')->paginate(PAGINATION_COUNT);
    }

    public function create()
    {
        return view('admin.users.create');
    }

    public function store($request)
    {

        $request = $this->handle_request($request);
        if (isset($request['interests'])) {
            $request['interests'] = implode(',', $request['interests']);
        }
        if (isset($request['tours_type'])) {
            $request['tours_type'] = implode(',', $request['tours_type']);
        }
        $request['day_price'] = isset($request['day_price']) ? (int)$request['day_price'] : 0;
        $request['hour_price'] = isset($request['hour_price']) ? (int)$request['hour_price'] : 0;
        $request['country_id'] = isset($request['country_id']) ? $request['country_id'] : null;
        $request['iban'] = isset($request['iban']) ? $request['iban'] : null;
		$request['logo'] = isset($request['logo']) ? uploadIamge($request['logo'], $this->crudName()) : null;
		$request['documented_files'] = isset($request['documented_files']) ? uploadIamges($request['documented_files'], $this->crudName()) : null;
        $user = $this->model->create($request);
        $days = [];
        if (isset($request['days']) && count($request['days']) > 0) {
            $days = [];
            foreach ($request['days'] as $day) {
                $days [] = [
                    'created_at' => now(),
                    'updated_at' => now(),
                    'day' => $day['day'], 
                    'day_status' => $day['day_status'] ?? null, 
                    'end' => $day['end'] ? Carbon::createFromFormat('H:i', $day['end'])->format('h:i A') : null, 
                    'start' => $day['start'] ? Carbon::createFromFormat('H:i', $day['start'])->format('h:i A') : null, 
                    'guide_id' => $user->id,
                ];
            }
        }
        if (count($days) > 0) {
            $user->days()->insert($days);
        }
        if (isset($request['languages'])) {
            $user->languages()->sync($request['languages']);
        }
        return $user;
    }

    public function edit($id)
    {
        $user = $this->findOne($id);
        return view('admin.users.update', compact('user'));
    }

    public function customUpdate($request, $id)
    {
        $user = $this->model->where("id", $id)->first();
        $request = $this->handle_update_request($request);
        if (isset($request['interests'])) {
            $request['interests'] = implode(',', $request['interests']);
        }
        if (isset($request['tours_type'])) {
            $request['tours_type'] = implode(',', $request['tours_type']);
        }
        $request['day_price'] = isset($request['day_price']) ? (int)$request['day_price'] : 0;
        $request['hour_price'] = isset($request['hour_price']) ? (int)$request['hour_price'] : 0;
        $request['country_id'] = isset($request['country_id']) ? $request['country_id'] : null;
        $request['iban'] = isset($request['iban']) ? $request['iban'] : null;
		$request['logo'] = isset($request['logo']) ? uploadIamge($request['logo'], $this->crudName()) : null;
		$request['documented_files'] = isset($request['documented_files']) ? uploadIamges($request['documented_files'], $this->crudName()) : null;
        $user->update($request);
        if (isset($request['days']) && count($request['days']) > 0) {
            $days = [];
            foreach ($request['days'] as $day) {
                $days [] = [
                    'created_at' => now(),
                    'updated_at' => now(),
                    'day' => $day['day'], 
                    'day_status' => $day['day_status'] ?? null,
                    'end' => $day['end'] ? Carbon::createFromFormat('H:i', $day['end'])->format('h:i A') : null, 
                    'start' => $day['start'] ? Carbon::createFromFormat('H:i', $day['start'])->format('h:i A') : null, 
                    'guide_id' => $user->id,
                ];
            }
            $user->days()->delete();
            $user->days()->insert($days);
        }
        if (isset($request['languages'])) {
            $user->languages()->sync($request['languages']);
        }
        return $user;
    }

    public function guideAlbum($request, $id)
    {
        $user = $this->model->where("id", $id)->first();
        if (isset($request->album) && count($request->album) > 0) {
            $album = []; 
            foreach ($request->album as $file) {
                if (!$file == null) {
                    $filePath = uploadIamge($file, $this->crudName()); // function on helper file to upload file
                    $album [] = [
                        'file' => $filePath,
                        'guide_id' => $id
                    ];
                }
            }
            DB::table('guide_album')->insert($album);
        }
        return $user;
    }

    public function archivesPage($offset, $limit)
    {
        $users = $this->archives($offset, $limit);
        return view('admin.users.archives', compact('users'));
    }

    public function handle_update_request($request)
    {
        $request->password ? $request->merge(['password' => bcrypt($request->password)]) : "";
        if (!$request->hasFile('file') == null) {
            $file = uploadIamge($request->file('file'), $this->crudName()); // function on helper file to upload file
            $request->merge(['img' => $file]);
        }
        if (!$request->hasFile('files') == null) {
            $files = uploadIamges($request->file('files'), $this->crudName()); // function on helper file to upload file
            $request->merge(['imgs' => $files]);
        }
        $request = array_filter(array_intersect_key($request->all(), $this->model->updateFildes()));
        return $request;
    }

    public function orders($offset, $limit)
    {
        return $this->guideOrder->with($this->guideOrder->model_relations())->unArchive()->orderBy('id', 'DESC')->paginate(PAGINATION_COUNT);
    }

}